*
* SOCKETS INTERFACE FOR MARINA
* 25 MAY 15 - D. FINNIGAN
*
SODYNPORTLO EQU 49152 ; LOWEST DYNAMIC TCP/UDP PORT
SOMAX EQU 4 ; MAX NUMBER OF OPEN SOCKETS
SOTYPEUDP EQU 1 ; SOCKET TYPE = UDP
SOTYPETCP EQU 2 ; SOCKET TYPE = TCP
SOTYPERAW EQU 4 ; SOCKET TYPE = RAW
SOCONN EQU 128 ; SOCKET IS CONNECTED (HI-BIT SET)
*
* SOCKET ERRORS
SONOSOCK EQU 1 ; SOCKET ID DOES NOT EXIST
SOINVARG EQU 2 ; INVALID ARGUMENT
SONOBUFS EQU 3 ; SOCKETS TABLE IS FULL
SOPORTUSED EQU 4 ; LOCAL PORT ALREADY IN USE
SOISCONN EQU 5 ; SOCKET ALREADY CONNECTED
SOADRNOTAVAIL EQU 6 ; ADDRESS/PORT NOT AVAILABLE
SONOTCONN EQU 7 ; SOCKET IS NOT CONNECTED
SOOPNOTSUPP EQU 8 ; OPERATION NOT SUPPORTED
*
* SOCKET OPTIONS/FLAGS
SORDYTOREAD EQU 128 ; SOCKET HAS DATA TO BE READ
SOREUSEPORT EQU 64 ; ALLOW DUPLICATE PORT BINDINGS
SOTYPE EQU 32 ; GET TYPE OF SOCKET (RD ONLY)
SOERROR EQU 16 ; GET AND CLEAR ERROR (RD ONLY)
*
*
* API ENTRY POINTS:
* ALL THESE SUBROUTINES USE PTR TO POINT TO A BLOCK OF
* DATA CONTAINING THE PARAMETERS.
*
*
* SOCKET
* CREATES A NEW SOCKET AND RETURNS SOCKET ID ON SUCCESS
* INPUT: SOCKET TYPE
* OUTPUT: SOCKET ID
SOCKET
 LDY #0 ; OFFSET FOR PTR
 LDA (PTR),Y ; CHECK SOCKET TYPE
 CMP #SOTYPEUDP ; IS IT UDP?
 BEQ :TYPEOK
 CMP #SOTYPETCP ; TCP?
 BNE :BADTYPE ; NOT AN ACCEPTABLE TYPE
*
* SOCKET TYPE IS OK, NOW CHECK FOR A FREE TABLE ENTRY
*
:TYPEOK
 LDX #SOMAX-1
:L LDA SOTABID,X
 BEQ :FOUNDENTRY ; NOT IN USE
 DEX
 BPL :L
* NO FREE ENTRIES
 LDA #SONOBUFS
 SEC
 RTS
* BAD SOCKET TYPE
:BADTYPE
 LDA #SOINVARG
 SEC
 RTS
* STORE SOCKET TYPE
:FOUNDENTRY
 LDA (PTR),Y
 STA SOTABTYPE,X
* GET NEXT SOCKET ID
 TXA
 PHA
 JSR SOGETNEXTID
 PLA ; GET SAVED X
 TAX
 LDA SONEXTID ; OUR NEW SOCKET ID
 STA SOTABID,X
* CLEAR OUT ANY OLD DATA IN THIS ENTRY SLOT
 LDA #0
 STA SOTABLPTL,X
 STA SOTABLPTH,X
 STA SOTABOPT,X
 STA SOTABFPTL,X
 STA SOTABFPTH,X
 STA SOTABERR,X
 TXA
 PHA ; SAVE X
 ASL
 ASL
 TAX
 LDY #0 ; ADDRESS COUNTER
 LDA #0
:L2 STA SOTABDEST,X ; CLEAR DESTINATION ADDRESS
 INX
 INY
 CPY #4
 BNE :L2
* SOCKET CREATED
 PLA
 TAX
 LDA SOTABID,X
 LDY #1 ; OFFSET FOR SOCKET ID IN RESULT
 STA (PTR),Y
 CLC
 RTS
*
*
* CONNECT
* ESTABLISH DESTINATION ADDRESS AND PORT FOR A SOCKET
* A TCP SOCKET CAN ONLY BE CONNECTED ONCE.
* IF FOREIGN ADDR IS 0, SOCKET WILL BE DISCONNECTED.
* INPUTS: SOCKET ID, FOREIGN ADDRESS, FOREIGN PORT
CONNECT
 LDY #0
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
* CHECK IF SOCKET IS ALREADY CONNECTED
 LDA SOTABTYPE,X
 AND #SOCONN
 BEQ :NOTCONN ; NOT CONNECTED
* SOCKET ALREADY CONNECTED
* ALLOW UDP SOCKETS TO REASSIGN ADDR/PORT
 LDA SOTABTYPE,X
 AND #%01111111 ; MASK OUT CONNECTED FLAG
 CMP #SOTYPEUDP ; IS IT UDP?
 BEQ :NOTCONN ; YES, ALLOW A RECONNECT
* CANNOT RECONNECT THIS SOCKET
 LDA #SOISCONN
 SEC
 RTS
* INVALID SOCKET
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
* VERIFY FOREIGN ADDR/PORT. BOTH MUST BE NON-ZERO
:NOTCONN
 LDY #5 ; LO-BYTE OF FOREIGN PORT
 LDA (PTR),Y
 INY
 ORA (PTR),Y
 BNE :CHECKADDR ; FOREIGN PORT IS OK
* FOREIGN PORT WAS LEFT UNDEFINED
 LDA #0
 STA SOTABFPTL,X
 STA SOTABFPTH,X
 LDA SOTABTYPE,X
 AND #%01111111 ; MARK SOCKET UNCONNECTED
 STA SOTABTYPE,X
 LDA #SOADRNOTAVAIL
 SEC
 RTS
:CHECKADDR
 LDY #1 ; POINT TO FOREIGN ADDRESS
 LDA (PTR),Y
 INY
 ORA (PTR),Y
 INY
 ORA (PTR),Y
 INY
 ORA (PTR),Y
 BNE :CHECKINUSE ; FOREIGN ADDRESS OK
* NO FOREIGN ADDRESS
 LDA SOTABTYPE,X
 AND #%01111111 ; MARK SOCKET UNCONNECTED
 STA SOTABTYPE,X
 LDA #SOADRNOTAVAIL
 SEC
 RTS
:CHECKINUSE
* NOW COMES THE REAL WORK: TO SEE IF THIS SET OF
* PORTS AND ADDRESSES IS ALREADY IN USE.
* IF OUR LOCAL PORT IS 0, THEN NO OTHER SOCKET WILL
* MATCH BECAUSE NO CONNECTED SOCKET WILL EVER HAVE A
* LOCAL PORT OF 0.
 LDA SOTABLPTL,X
 ORA SOTABLPTH,X
 BEQ :BIND  ; LOCAL PORT IS UNBOUND
* IF WE GET HERE, A LOCAL PORT HAS BEEN SET SO WE NEED
* TO DO A FULL CHECK ON THE SOCKETS TABLE.
 NOP
 JMP :SETFOREIGN
:BIND
* CALL BIND2  TO PICK A LOCAL PORT
 JSR BIND2
* SET FOREIGN PORT AND ADDRESS IN SOCKET TABLE
:SETFOREIGN
 LDY #5 ; LO-BYTE OF FOREIGN PORT
 LDA (PTR),Y
 STA SOTABFPTL,X
 INY
 LDA (PTR),Y
 STA SOTABFPTH,X
 LDY #1 ; POINT TO FOREIGN ADDRESS
 TXA
 PHA
 ASL
 ASL  ; COMPUTE TABLE OFFSET
 TAX
:FAL LDA (PTR),Y
 STA SOTABDEST,X ; COPY FOREIGN ADDRESS
 INX
 INY
 CPY #5
 BNE :FAL
* SET STATUS TO CONNECTED
 PLA
 TAX
 LDA SOTABTYPE,X
 ORA #SOCONN ; SET CONNECTED FLAG
 STA SOTABTYPE,X
* DONE
 CLC ; NO ERROR
 RTS
*
*
* SENDTO
* SEND DATA ON A SOCKET
* INPUTS: SOCKET ID, DEST ADDR, DEST PORT, BUFFER, DATA LEN
SENDTO
 LDY #0
 LDA (PTR),Y ; GET SOCKET ID
 JSR SOGETSOBYID
 BCC :CHECKSOERR
* INVALID SOCKET
 LDA #SOINVARG
 SEC
 RTS
* CHECK FOR SOCKET ERROR
:CHECKSOERR
 LDA SOTABERR,X
 BEQ :SEND3  ; NO ERROR
 PHA
 LDA #0
 STA SOTABERR,X
 PLA
 SEC
 RTS
* COPY OUT PARAMETERS
:SEND3
 LDY #7 ; LO-BYTE OF BUFFER
 LDA (PTR),Y
 STA OUTPBUF
 INY
 LDA (PTR),Y
 STA OUTPBUF+1
 INY  ; LO-BYTE OF DATA LENGTH
 LDA (PTR),Y
 STA OUTPLEN
 INY
 LDA (PTR),Y
 STA OUTPLEN+1
* NOW WE NEED TO SET THE IP PARAMETERS
 LDA #0
 STA OUTPHEAD+18 ; IDENTIFICATION
 STA OUTPHEAD+19
 STA OUTPHEAD+22 ; TTL
 LDA #1
 STA OUTPHEAD+26 ; SOURCE ADDR
* COPY DESTINATION ADDRESS
 TXA
 PHA
 LDX #30 ; DEST IP ADDR OFFSET
 LDY #1 ; POINT TO DEST ADDR IN PARAM
:FAL LDA (PTR),Y
 STA OUTPHEAD,X
 INY
 INX
 CPX #34
 BNE :FAL
* COPY UDP PORTS
 PLA
 TAX
* DO WE HAVE A LOCAL PORT DEFINED?
 LDA SOTABLPTL,X
 ORA SOTABLPTH,X
 BNE :COPYLOCAL ; YES
* NO, SO CALL BIND2 TO SET LOCAL PORT
 JSR BIND2
* SET LOCAL PORT
:COPYLOCAL
 LDA SOTABLPTH,X
 STA OUTPHEAD+34
 LDA SOTABLPTL,X
 STA OUTPHEAD+35
* SET FOREIGN PORT
 LDA (PTR),Y
 STA OUTPHEAD+37
 INY
 LDA (PTR),Y
 STA OUTPHEAD+36
* AND WE'RE DONE HERE
 JSR UDPSEND
* SHOULD CHECK FOR ERRORS
 RTS
*
*
* SEND
* SEND DATA ON A CONNECTED SOCKET
* INPUTS: SOCKET ID, BUFFER, DATA LEN
SEND
 LDY #0
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
* SOCKET MUST BE CONNECTED
 LDA SOTABTYPE,X
 AND #SOCONN
 BNE :SEND2
* SOCKET NOT CONNECTED
 LDA #SONOTCONN
 SEC
 RTS
* INVALID SOCKET
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
* CHECK FOR SOCKET ERROR
:SEND2
 LDA SOTABERR,X
 BEQ :SEND3  ; NO ERROR
 PHA
 LDA #0
 STA SOTABERR,X
 PLA
 SEC
 RTS
* COPY OUT PARAMETERS
:SEND3
 LDY #1 ; LO-BYTE OF SEND BUFFER
 LDA (PTR),Y
 STA OUTPBUF
 INY
 LDA (PTR),Y
 STA OUTPBUF+1
 INY  ; LO-BYTE OF DATA LENGTH
 LDA (PTR),Y
 STA OUTPLEN
 INY
 LDA (PTR),Y
 STA OUTPLEN+1
* NOW WE NEED TO SET THE IP PARAMETERS
 LDA #0
 STA OUTPHEAD+18 ; IDENTIFICATION
 STA OUTPHEAD+19
 STA OUTPHEAD+22 ; TTL
 LDA #1
 STA OUTPHEAD+26 ; SOURCE ADDR
* COPY DESTINATION ADDRESS
 TXA
 PHA ; STORE SOCKET TABLE OFFSET
 ASL
 ASL
 TAX ; FOREIGN ADDRESS OFFSET
 LDY #30 ; DEST IP ADDR OFFSET
:FAL LDA SOTABDEST,X
 STA OUTPHEAD,Y
 INX
 INY
 CPY #34
 BNE :FAL
* COPY UDP PORTS
 PLA
 TAX ; GET SOCKET TABLE OFFSET BACK
 LDA SOTABLPTH,X
 STA OUTPHEAD+34
 LDA SOTABLPTL,X
 STA OUTPHEAD+35
 LDA SOTABFPTH,X
 STA OUTPHEAD+36
 LDA SOTABFPTL,X
 STA OUTPHEAD+37
* AND WE'RE DONE HERE
 JSR UDPSEND
* SHOULD CHECK FOR ERRORS
 RTS
*
*
* RECVFROM
* RECEIVE DATA ON A SOCKET
* RETURNS ADDR/PORT OF SENDER
* INPUTS: SOCKET ID
* OUTPUTS: BUFFER, LEN, FOREIGN ADDRESS, FOREIGN PORT
RECVFROM
 LDY #0
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCC :CHECKSOERR
* INVALID SOCKET
 LDA #SOINVARG
 SEC
 RTS
* CHECK FOR SOCKET ERROR
:CHECKSOERR
 LDA SOTABERR,X
 BEQ :CHECKREADY
 PHA
 LDA #0
 STA SOTABERR,X
 PLA
 SEC
 RTS
* CHECK IF THIS SOCKET IS READY TO READ
:CHECKREADY
 LDA SOTABOPT,X
 AND #SORDYTOREAD
 BNE :READYTOREAD
* SOCKET NOT READY TO READ
 LDA #0 ; CLEAR BUFFER AND LENGTH
 LDY #1 ; OFFSET IN PARAMETER BLOCK
:CPL STA (PTR),Y
 INY
 CPY #5
 BNE :CPL
 SEC
 RTS
* SOCKET IS READY TO READ. WE NEED TO GET THE BUFFER ADDR
* AS WELL AS DATA LENGTH.
:READYTOREAD
 LDY #1 ; LO-BYTE OF BUFFER ADDR
 CLC
 LDA INPBUF
 ADC #8 ; SKIP UDP HEAD
 STA (PTR),Y
 INY
 LDA INPBUF+1
 ADC #0 ; SKIP UDP HEAD
 STA (PTR),Y
* DATA LENGTH
* ASSUMES UDP FOR NOW
* NOTE THAT UDPDATLEN DOES NOT COUNT THE UDP HEADER!
 INY
 LDA UDPDATLEN
 STA (PTR),Y
 INY
 LDA UDPDATLEN+1
 STA (PTR),Y
* COPY FOREIGN ADDRESS
 TXA
 PHA  ; SAVE SOCKET OFFSET
 LDY #8 ; OFFSET IN PARAMETER BLOCK
 LDX #3
:CSL LDA IPINSRC,X
 STA (PTR),Y
 DEY
 DEX
 BPL :CSL
* COPY FOREIGN PORT
 LDY #9
 LDA UDPFPORT
 STA (PTR),Y
 INY
 LDA UDPFPORT+1
 STA (PTR),Y
* CLEAR THE READY TO READ FLAG
 PLA
 TAX
 LDA SOTABOPT,X
 AND #%01111111
 STA SOTABOPT,X
 CLC  ; NO ERROR
 RTS
*
*
* RECV
* RECEIVE DATA ON A CONNECTED SOCKET
* INPUTS: SOCKET ID
* OUTPUT: BUFFER, LENGTH
RECV
 LDY #0
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
* SOCKET MUST BE CONNECTED
 LDA SOTABTYPE,X
 AND #SOCONN
 BNE :CHECKSOERR
* SOCKET NOT CONNECTED
 LDA #SONOTCONN
 SEC
 RTS
* INVALID SOCKET
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
* CHECK FOR SOCKET ERROR
:CHECKSOERR
 LDA SOTABERR,X
 BEQ :CHECKREADY
 PHA
 LDA #0
 STA SOTABERR,X
 PLA
 SEC
 RTS
* CHECK IF THIS SOCKET IS READY TO READ
:CHECKREADY
 LDA SOTABOPT,X
 AND #SORDYTOREAD
 BNE :READYTOREAD
* SOCKET NOT READY TO READ
 LDA #0
 LDY #1 ; OFFSET IN PARAMETER BLOCK
:CPL STA (PTR),Y
 INY
 CPY #5
 BNE :CPL
 SEC
 RTS
* SOCKET IS READY TO READ. WE NEED TO GET THE BUFFER ADDR
* AS WELL AS DATA LENGTH.
:READYTOREAD
 LDY #1 ; LO-BYTE OF BUFFER ADDR
 CLC
 LDA INPBUF
 ADC #8 ; SKIP UDP HEAD
 STA (PTR),Y
 INY
 LDA INPBUF+1
 ADC #0 ; SKIP UDP HEAD
 STA (PTR),Y
* DATA LENGTH
* ASSUMES UDP FOR NOW
* NOTE THAT UDPDATLEN DOES NOT COUNT THE UDP HEADER!
 INY
 LDA UDPDATLEN
 STA (PTR),Y
 INY
 LDA UDPDATLEN+1
 STA (PTR),Y
* CLEAR THE READY TO READ FLAG
 LDA SOTABOPT,X
 AND #%01111111
 STA SOTABOPT,X
 CLC  ; NO ERROR
 RTS
*
*
* BIND
* ASSIGN LOCAL PORT NUMBER TO A SOCKET
* INPUTS: SOCKET ID, LOCAL PORT.
* LEAVE LOCAL PORT 0 TO HAVE A DYNAMIC PORT AUTOMATICALLY SET.
*
* CLARIFICATION ON LOCAL PORT:
* LOCAL PORT NUMBERS CAN BE REUSED AS LONG AS THE REMOTE
* ENDPOINTS DIFFER. THE CHECK IS MADE WITH A 4-WAY MATCH:
* SOCKET TYPE (UDP OR TCP), REMOTE ADDRESS, AND
* LOCAL AND REMOTE PORT NUMBER. IF ANOTHER SOCKET MATCHES
* ALL 4 OF THESE CRITERIA, THEN THE BIND OPERATION FAILS.
*
* IN THE CASE OF MARINA, THERE CAN ONLY EVER BE 1 LOCAL
* ADDRESS, SO THE ADDRESS CHECK IS SIMPLIFIED TO ONLY
* LOOK AT THE REMOTE ADDRESS.
*
BIND
 LDY #0 ; PTR OFFSET
 LDA (PTR),Y ; GET SOCKET ID
 JSR SOGETSOBYID
 BCS :BADSOCK ; NO VALID SOCKET
* CHECK IF SOCKET IS ALREADY BOUND ON LOCAL PORT
 LDA SOTABLPTL,X
 ORA SOTABLPTH,X
 BEQ :NEXT ; NOT BOUND, CONTINUE
* SOCKET ALREADY BOUND
 LDA #SOINVARG
 SEC
 RTS
* NOT A VALID SOCKET
:BADSOCK
 LDA #SONOSOCK
 SEC
 RTS
* CHECK IF LOCAL PORT WAS DEFINED
:NEXT
 LDY #1 ; LOW-BYTE OF LOCAL PORT
 LDA (PTR),Y
 INY
 ORA (PTR),Y
 BEQ :GETDYN ; CHOOSE A DYNAMIC PORT
* AT THIS POINT, USER HAS CHOSEN A PORT. WE NEED TO CHECK IF IT
* IS IN USE. IF SO, PROCEED TO NEXT CHECK.
* IF NOT, COMPLETE BINDING.
 TXA
 PHA ; SAVE X, SOCKET OFFSET
 LDX #SOMAX-1 ; NUMBER OF ENTRIES TO SCAN
:L LDY #1
 LDA (PTR),Y ; LO-BYTE OF LOCAL PORT
 CMP SOTABLPTL,X
 BNE :NEXTSOCK ; NO MATCH, CHECK NEXT SOCKET
 INY
 LDA (PTR),Y ; HI-BYTE
 CMP SOTABLPTH,X
 BNE :NEXTSOCK
* IF WE GET HERE, THIS LOCAL PORT IS IN USE.
* CHECK IF REMOTE PORT MATCHES TOO.
 PLA ; PULL X OFF STACK
 LDA #SOPORTUSED
 SEC
 RTS
:NEXTSOCK
 DEX
 BPL :L ; CHECK NEXT SOCKET
* LOCAL PORT IS FREE TO USE
 PLA ; GET OLD X BACK
 TAX ; SOCKET TABLE OFFSET
 LDY #1 ; LOW BYTE OF LOCAL PORT
 LDA (PTR),Y
 STA SOTABLPTL,X
 INY
 LDA (PTR),Y
 STA SOTABLPTH,X
 CLC  ; NO ERROR
 RTS  ; DONE!
:GETDYN
* USER DID NOT SPECIFY A PORT, SO WE PICK ONE FOR HIM
* BIND2 IS ALTERNATE ENTRY POINT FOR CONNECT. DO NOT USE!
BIND2
 TXA
 PHA
 JSR SOGETNEXTPORT ; LEAVES RESULT IN SONEXTPORT
 PLA
 TAX
 LDA SONEXTPORT
 STA SOTABLPTL,X
 LDA SONEXTPORT+1
 STA SOTABLPTH,X
 CLC  ; NO ERROR
 RTS
*
*
* LISTEN
* ALLOW A BOUND SOCKET TO RECEIVE INCOMING TCP CONNECTIONS
* INCOMING QUEUE BACKLOG IS NOT IMPLEMENTED
* INPUT: SOCKET ID
LISTEN
 LDY #0 ; POINT TO SOCKET ID
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
*
* SOCKET EXISTS. CHECK TO MAKE SURE IT'S TCP
*
 LDA SOTABTYPE,X ; GET SOCKET TYPE
 TAY  ; SAVE FOR LATER
 AND #SOTYPETCP ; IS IT TCP?
 BEQ :BADTYPE ; YES, SO MOVE ON
*
* SOCKET IS TCP. CHECK IF IT'S ALREADY CONNECTED
*
 TYA  ; GET SOCKET TYPE BACK
 AND #SOCONN ; IS THE SOCKET CONNECTED?
 BNE :ISCONN ; ALREADY CONNECTED
*
* WRONG SOCKET TYPE
*
:BADTYPE
 LDA #SOOPNOTSUPP ; OPERATION NOT SUPPORTED
 SEC
 RTS
*
* SOCKET DID NOT EXIST
*
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
*
* SOCKET IS ALREADY CONNECTED
*
:ISCONN
 LDA #SOINVARG
 SEC
 RTS
*
*
* GETSOCKNAME
* RETURN THE LOCAL ADDRESS AND PORT FOR A SOCKET.
* INPUT: SOCKET ID
* OUTPUT: ADDRESS, PORT
GETSOCKNAME
 LDY #0 ; POINT TO SOCKET ID
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
* SOCKET EXISTS, SO COPY LOCAL INFO
 LDY #4  ; OFFSET TO STORE LOCAL ADDR
 TXA
 PHA
 LDX #3 ; OFFSET FOR IPADDR
:ADL LDA IPADDR,X
 STA (PTR),Y
 DEY
 DEX
 BPL :ADL
* NOW GET LOCAL PORT
 PLA ; GET X BACK
 TAX
 LDA SOTABLPTL,X ; LO-BYTE OF LOCAL PORT
 LDY #5 ; OFFSET FOR LO-BYTE
 STA (PTR),Y
 INY
 LDA SOTABLPTH,X ; HI-BYTE
 STA (PTR),Y
 CLC  ; NO ERROR
 RTS
* SOCKET DID NOT EXIST
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
*
*
* SCLOSE
* CLOSE A SOCKET
* INPUT: SOCKET ID
SCLOSE
 LDY #0
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
* MARK SOCKET AS CLOSED
 LDA #0
 STA SOTABID,X
 STA SOTABTYPE,X
 CLC
 RTS
* INVALID SOCKET
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
*
*
* GETSOCKOPT
* GET OPTIONS FOR A SOCKET
* INPUT: SOCKET ID, OPTION NAME
* OUTPUT: OPTION VALUE
*
GETSOCKOPT
 LDY #0
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
*
* SOCKET ID IS OK, NOW GET OPTION.
* AT PRESENT, ALL OPTIONS ARE MERELY BIT FLAGS, EXCEPT
* FOR TYPE AND ERROR.
*
 INY ; POINT TO DESIRED OPTION
*
* CHECK FOR THE TYPE OR ERROR OPTIONS
*
 LDA (PTR),Y
 CMP #SOTYPE ; RETURN SOCKET TYPE
 BEQ :GETTYPE
 CMP #SOERROR ; RETURN SOCKET ERROR
 BEQ :GETERROR
*
* IF WE GET HERE, IT'S A NORMAL OPTION
*
 LDA SOTABOPT,X ; GET OPTIONS BYTE FOR THIS SOCKET
 AND (PTR),Y ; MASK DESIRED OPTION
 BEQ :STORE ; OPTION IS NOT SET
 LDA #$FF ; OPTION IS SET
:STORE
 INY  ; STORE THE RESULT
 STA (PTR),Y
 CLC  ; NO ERROR
 RTS
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
:GETTYPE
 LDA SOTABTYPE,X
 AND #%01111111 ; MASK OUT CONNECTED FLAG
 BNE :STORE ; ALWAYS TAKEN
:GETERROR
 LDA SOTABERR,X
 JMP :STORE
*
*
* SETSOCKOPT
* SET OPTIONS FOR A SOCKET
* INPUT: SOCKET ID, OPTION NAME, OPTION VALUE
* OUTPUT: NOTHING
*
SETSOCKOPT
 LDY #0
 LDA (PTR),Y
 JSR SOGETSOBYID
 BCS :BADSOCK
*
* CHECK FOR THE TWO READ-ONLY OPTIONS
*
 INY
 LDA (PTR),Y
 CMP #SOTYPE
 BEQ :BADSOCK ; USED FOR INVALID ARG
 CMP #SOERROR
 BEQ :BADSOCK ; INVALID ARGUMENT
*
* GET OPTION VALUE. AT PRESENT, ALL OPTIONS ARE BIT FLAGS.
*
 INY
 LDA (PTR),Y
 BEQ :CLEARFLAG
*
* SET THE OPTION FLAG
*
 DEY ; BACK TO OPTION NAME
 LDA SOTABOPT,X
 ORA (PTR),Y ; SET IT
 CLC
 RTS
*
* CLEAR THE OPTION FLAG
*
:CLEARFLAG
 DEY  ; BACK TO OPTION NAME
 LDA SOTABOPT,X
 EOR (PTR),Y ; CLEAR IT
 STA SOTABOPT,X
 CLC
 RTS
:BADSOCK
 LDA #SOINVARG
 SEC
 RTS
*
*
*
* BACKEND SUBROUTINES -- THESE ARE INTERNAL TO THE API.
* AND STORAGE TOO.
*
* SOCKETS TABLE
SOTABID DS 1*SOMAX ; SOCKET IDS (1 BYTE)
SOTABTYPE DS 1*SOMAX ; SOCKET TYPE + CONN. STATUS (1 BYTE)
SOTABLPTL DS 1*SOMAX ; LOCAL PORT (LO-BYTE)
SOTABLPTH DS 1*SOMAX ; LOCAL PORT (HI-BYTE)
SOTABOPT DS 1*SOMAX ; SOCKET OPTIONS (1 BYTE)
SOTABDEST DS 4*SOMAX ; FOREIGN ADDRESS (4 BYTES)
SOTABFPTL DS 1*SOMAX ; FOREIGN PORT (LO-BYTE)
SOTABFPTH DS 1*SOMAX ; FOREIGN PORT (HI-BYTE)
SOTABERR DS 1*SOMAX ; ICMP ERROR (1 BYTE)
*
*
* GET NEXT LOCAL PORT NUMBER
* EVEN THOUGH TCP AND UDP PORT NUMBERS NEVER INTERFERE,
* THIS ROUTINE WILL USE A SINGLE NUMBER SOURCE FOR BOTH.
SONEXTPORT DS 2 ; NEXT DYNAMIC PORT NUMBER
SOGETNEXTPORT
 LDA SONEXTPORT
 ORA SONEXTPORT+1
 BEQ :ROLLOVER ; INITIALIZE PORT COUNTER
 INC SONEXTPORT
 BNE :TOTAL
 INC SONEXTPORT+1
 BEQ :ROLLOVER
:TOTAL LDX #SOMAX-1 ; NUMBER OF ENTRIES TO SCAN
:CHECKSOCK
* SEE IF SONEXTPORT IS IN USE BY ANY ACTIVE SOCKET
 LDA SOTABID,X ; IS THIS SOCKET IN USE?
 BEQ :NEXTSOCK ; NO, SKIP IT
 LDA SOTABLPTH,X
 CMP SONEXTPORT
 BNE :NEXTSOCK ; PORT DOESN'T MATCH, SKIP
 LDA SOTABLPTL,X
 CMP SONEXTPORT+1
 BNE :NEXTSOCK
* IF WE GOT HERE, THIS PORT IS ALREADY IN USE
* SO INCREMENT SONEXTPORT AND CHECK AGAIN.
 INC SONEXTPORT
 BNE :NEXTSOCK
 INC SONEXTPORT+1
 BNE :NEXTSOCK ; USUALLY WILL BE TAKEN
* A RARE CASE: PORT NUMBER ROLLOVER
:ROLLOVER
 LDA #<SODYNPORTLO
 STA SONEXTPORT
 LDA #>SODYNPORTLO
 STA SONEXTPORT+1
 JMP :TOTAL ; START ALL OVER
:NEXTSOCK
 DEX
 BPL :CHECKSOCK
* IF WE MADE IT HERE, THIS PORT SHOULD BE FREE TO USE
 CLC ; NO ERROR
 RTS
*
* GET NEXT SOCKET ID
* INPUT: NONE. RETURNS NEXT SOCKET ID IN A-REG.
SONEXTID DFB 0
SOGETNEXTID
 INC SONEXTID
 BNE :NOROLL
 LDA #1
 STA SONEXTID
:NOROLL
 LDX #SOMAX-1
:L LDA SOTABID,X
 CMP SONEXTID
 BNE :NEXT
* ID IN USE
 INC SONEXTID
 BNE :NEXT
 LDA #1
 STA SONEXTID
 BNE :NOROLL ; ALWAYS TAKEN
:NEXT DEX
 BPL :L
* ID IS NOT IN USE
 LDA SONEXTID
 CLC  ; NO ERROR
 RTS
*
* GET A UDP SOCKET BY TYPE, ADDRESS, AND PORT
* THIS IS USED BY INCOMING DATAGRAM PROCESSING.
* INPBUF MUST POINT TO START OF UDP HEAD.
SOGETUDPSOBY
 LDX #SOMAX-1 ; NUMBER OF ENTRIES TO SCAN
:CHECKSOCK
 LDA SOTABID,X ; IS THIS SOCKET IN USE?
 BNE :N ; YES, CONTINUE
 JMP :NEXTSOCK ; NO, SKIP IT
:N
 LDA SOTABTYPE,X ; CHECK SOCKET TYPE
 AND #%01111111 ; MASK OUT CONNECTED FLAG
 CMP #SOTYPEUDP ; IS IT A UDP SOCKET?
 BEQ :N2
 JMP :NEXTSOCK ; NO, SO SKIP IT
:N2
* NOW WE'VE ESTABLISHED THIS IS AN ACTIVE UDP SOCKET,
* WE NEED TO CHECK THE LOCAL PORT.
 LDY #2 ; OFFSET OF DEST PORT
 LDA (INPBUF),Y ; GET HI-BYTE UDP DEST PORT
 CMP SOTABLPTH,X ; CHECK SOCKET'S LOCAL PORT
 BNE :NEXTSOCK ; PORT MISMATCH, SKIP IT
 INY  ; POINT TO LO-BYTE UDP DEST PORT
 LDA (INPBUF),Y ; GET LO-BYTE OF DEST PORT
 CMP SOTABLPTL,X
 BNE :NEXTSOCK ; PORT MISMATCH, SKIP IT
* AT THIS POINT, IF THE SOCKET IS NOT CONNECTED, WE HAVE
* FOUND THE MATCH.
 LDA SOTABTYPE,X ; CONNECTED STATUS IS HI-BIT
 AND #SOCONN ; ISOLATE CONNECTED FLAG
 BNE :ISCONN ; SOCKET IS CONNECTED, MORE CHECKS
* IF WE MAKE IT HERE, SOCKET IS NOT CONNECTED, SO WE ARE DONE.
 LDA SOTABID,X ; LOAD SOCKET ID
 CLC  ; NO ERROR
 RTS
:ISCONN
* SOCKET IS CONNECTED, SO WE NEED TO DO 2 MORE CHECKS
 LDY #0 ; OFFSET FOR SOURCE PORT
 LDA (INPBUF),Y ; GET HI-BYTE UDP SOURCE PORT
 CMP SOTABFPTH,X ; CHECK SOCKET'S FOREIGN PORT
 BNE :NEXTSOCK
 INY  ; POINT TO LO-BYTE OF UDP SRC PORT
 LDA (INPBUF),Y
 CMP SOTABFPTL,X
 BNE :NEXTSOCK
* NOW FOREIGN PORT MATCHES OUR SOURCE PORT, SO CHECK ADDRESS
 SEC ; MOVE INPBUF BACK TO IP HEAD
 LDA INPBUF
 SBC IPHEADLEN
 STA INPBUF
 LDA INPBUF+1
 SBC #0
 STA INPBUF+1
* GET SOCKET TABLE OFFSET FOR FOREIGN ADDRESS
 TXA
 PHA ; SAVE X FOR LATER
 ASL
 ASL  ; MULTIPLY BY 4
 TAX ; THIS IS SOTABDEST OFFSET
 LDY #12 ; SOURCE ADDRESS OFFSET
:FAL LDA (INPBUF),Y
 CMP SOTABDEST,X
 BNE :FANOMATCH ; ADDRESS DOES NOT MATCH
 INX
 INY
 CPY #16
 BNE :FAL
* IF WE MAKE IT HERE, THEN THE ADDRESSES MATCH.
 PLA ; GET X BACK
 TAX
 LDA SOTABID,X
 CLC  ; NO ERROR
 RTS
:FANOMATCH
 PLA
 TAX
:NEXTSOCK
 DEX  ; MOVE TO NEXT SOCKET ENTRY
 BPL :CHECKSOCK
:NOSOCKSMATCH
 SEC  ; ERROR, NO MATCHING SOCKETS
 RTS
*
*
* GET A TCP SOCKET BY TYPE, ADDRESS, AND PORT.
* THIS IS USED BY INCOMING SEGMENT PROCESSING.
* INPBUF MUST POINT TO START OF TCP HEAD.
*
* HERE IS AN OUTLINE OF THE ALGORITHM:
* 1.) LOOP OVER EACH SOCKET TABLE ENTRY
* 2.) IF THE ENTRY IS NOT IN USE OR IS NOT TCP,
*     SKIP IT.
* 3.) CHECK FOR A 4-WAY MATCH ON PORTS/ADDRESSES.
* 4.) IF NO MATCH, CHECK FOR LISTENING SOCKET AND
*     MATCH LISTENING PORT NUMBER.
* 5.) IF NO MATCHES FROM STEPS 3 AND 4, MOVE TO NEXT
*     SOCKET ENTRY.
* 6.) IF STEP 4 MATCHED, CHECK FOR SYN FLAG.
*
SOGETTCPSOBY
 LDX #SOMAX-1 ; NUMBER OF ENTRIES TO SCAN
:CHECKSOCK
 LDA SOTABID,X ; IS THIS SOCKET IN USE?
 BNE :N ; YES, CONTINUE
 JMP :NEXTSOCK ; NO, SKIP IT
:N
 LDA SOTABTYPE,X ; CHECK SOCKET TYPE
 AND #%01111111 ; MASK OUT CONNECTED FLAG
 CMP #SOTYPETCP ; IS IT A TCP SOCKET?
 BEQ :N2
 JMP :NEXTSOCK ; NO, SO SKIP IT
:N2
* NOW WE'VE ESTABLISHED THIS IS AN OPEN TCP SOCKET,
* WE NEED TO CHECK THE LOCAL PORT
 LDY #2 ; OFFSET OF DEST PORT
 LDA (INPBUF),Y ; GET HI-BYTE OF TCP DEST PORT
 CMP SOTABLPTH,X CHECK SOCKET'S LOCAL PORT
 BNE :NEXTSOCK ; PORT MISMATCH, SKIP IT
 INY ; POINT TO LO-BYTE OF TCP DEST PORT
 LDA (INPBUF),Y
 CMP SOTABLPTL,X
 BNE :NEXTSOCK
*
* NOW CHECK REMOTE PORT
*
:NEXTSOCK
 DEX ; MOVE TO NEXT SOCKET ENTRY
 BPL :CHECKSOCK
:NOSOCKMATCH
 SEC  ; ERROR, NO MATCHING SOCKETS
 RTS
*
*
* GET A SOCKET BY ID
* ID TO SEARCH FOR IN A-REG, RETURNS SOCKET OFFSET IN X, CARRY
* CLEAR; CARRY SET IF NOT FOUND.
SOGETSOBYID
 LDX #SOMAX-1
:L CMP SOTABID,X
 BEQ :FOUND
 DEX
 BPL :L
* SOCKET NOT FOUND
 SEC
 RTS
* SOCKET FOUND
:FOUND
 CLC
 RTS
